/* Library for quickly and easily executing regular expressions in C */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <regex.h>
#include <stdarg.h>

/* This buffer size is a shot in the dark :( */
#define ERRBUFSIZE 255

int re_checkreturn(int code, char* str, regex_t* preg)
{
	char errbuf[ERRBUFSIZE];

	/* Exit immediately if the condition is all clear */
	if ( code == 0 || code == REG_NOMATCH )
		return 0;


	/* An error occured */
	/* I should probably use regerror, but I have no idea what a good size
	 * for errbuf_size is
	 */
	fprintf(stderr, "regex failed: ");

	regerror(code, preg, errbuf, ERRBUFSIZE);

	fprintf(stderr, "%s: %s\n", errbuf, str);
	exit(1);
	return 1;
}


int regex(char* str, char* re, int compflags, int execflags, ...)
{
	va_list ap;
	char* arg;

	regex_t preg;
	size_t nmatch;
	int currmatch;
	regmatch_t* pmatch;
	int result;

	int length;
	int currchar;

	/* Compile the regular expression */
	result = regcomp(&preg, re, compflags);
	re_checkreturn(result, re, &preg);

	/* Dynamically compute how many elements pmatch must contain */
	nmatch = 1;
	if ( ! ( compflags & REG_NOSUB ) )
	{
		length = strlen(re);
		for ( currchar = 0; currchar < length; currchar++ )
		{
			if ( re[currchar] == '(' )
				nmatch++;
		}
	}

	if ( nmatch == 1 )
	{
		/* no subexpressions */
		nmatch = 0;
		pmatch = NULL;
	}
	else
	{
		pmatch = malloc(sizeof(regmatch_t) * nmatch);
		if ( pmatch == NULL )
		{
			perror("malloc");
			exit(1);
		}
	}

	/* Execute the regular expression */
	if ( (result = regexec(&preg, str, nmatch, pmatch, execflags)) != 0 )
	{
		re_checkreturn(result, re, &preg);
		regfree(&preg);
		if ( pmatch != NULL )
			free(pmatch);

		return 0;	/* No match */
	}

	/* If no-subexpression matching was specified, we can stop here and
	 * avoid all of the argument processing
	 */
	if ( compflags & REG_NOSUB )
	{
		regfree(&preg);
		return 1;
	}

	/* Copy the matched subexpressions (if any) into the varargs */
	va_start(ap, execflags);
	/* the 0th slot is reserved for the whole expression, since I don't
	 * care about it, skip it
	 */

	for ( currmatch = 1; currmatch < nmatch; currmatch++ )
	{
		arg = va_arg(ap, char*);
		if ( arg == NULL )
		{
			fprintf(stderr, "regex warning:  more parenthesized subexpressions than arguments to regex: %s\n", re);
			return 1;
		}

		memcpy(arg, str + pmatch[currmatch].rm_so, 
			pmatch[currmatch].rm_eo - pmatch[currmatch].rm_so);
		arg[pmatch[currmatch].rm_eo - pmatch[currmatch].rm_so] = '\0';
	}

	va_end(ap);
	regfree(&preg);
	free(pmatch);

	return 1;
}
